#include "save_image.h"

#include <iostream>

#include "jpeglib.h"
#include "png.h"

#ifdef _WIN32
#include <windows.h>

#include "specific_os.h"
#endif

#include "read_pdf.h"


std::vector<unsigned char> SaveImage::bufferBGRAToRGB(
    const unsigned char* bufferBGRA,
    const int& renderingWidthPixels,
    const int& renderingHeightPixels,
    const int& stride
    ) {
    std::vector<unsigned char> bufferRGB(renderingWidthPixels * renderingHeightPixels * 3);

    for (int y = 0; y < renderingHeightPixels; ++y) {
        for (int x = 0; x < renderingWidthPixels; ++x) {
            int sourceIndex = y * stride + x * 4;
            int destinationIndex = (y * renderingWidthPixels + x) * 3;
            bufferRGB[destinationIndex + 0] = bufferBGRA[sourceIndex + 2];
            bufferRGB[destinationIndex + 1] = bufferBGRA[sourceIndex + 1];
            bufferRGB[destinationIndex + 2] = bufferBGRA[sourceIndex + 0];
        }
    }

    return bufferRGB;
}

bool SaveImage::saveBufferToJPEG(
    const std::filesystem::path& pageImagePath,
    const int& renderingWidthPixels,
    const int& renderingHeightPixels,
    const int& imageQuality,
    const int& channels,
    const int& dpi,
    unsigned char* buff
    ) {
#ifdef _WIN32
    auto pageImagePathWcharString = SpecificOS::convertUTF8ToWchar(pageImagePath.string());
    FILE* file = _wfopen(pageImagePathWcharString.c_str(), L"wb");
#else
    FILE* file = fopen(pageImagePath.string().c_str(), "wb");
#endif

    jpeg_compress_struct cinfo{};
    jpeg_error_mgr jerr{};
    cinfo.err = jpeg_std_error(&jerr);
    jpeg_create_compress(&cinfo);

    jpeg_stdio_dest(&cinfo, file);
    cinfo.image_width = renderingWidthPixels;
    cinfo.image_height = renderingHeightPixels;
    cinfo.input_components = channels;

    if (channels == 3) {
        cinfo.in_color_space = JCS_RGB;
    } else if (channels == 1) {
        cinfo.in_color_space = JCS_GRAYSCALE;
    }

    jpeg_set_defaults(&cinfo);
    jpeg_set_quality(&cinfo, imageQuality, true);

    cinfo.density_unit = 1;
    cinfo.X_density = dpi;
    cinfo.Y_density = dpi;

    jpeg_start_compress(&cinfo, true);

    JSAMPROW row_pointer[1];
    int row_stride = renderingWidthPixels * channels;

    while (cinfo.next_scanline < cinfo.image_height) {
        row_pointer[0] = &buff[cinfo.next_scanline * row_stride];
        jpeg_write_scanlines(&cinfo, row_pointer, 1);
    }

    jpeg_finish_compress(&cinfo);

    fclose(file);
    jpeg_destroy_compress(&cinfo);

    if (std::filesystem::exists(pageImagePath)) {
        return true;
    }

    return false;
}

bool SaveImage::saveBufferToPNG(
    const std::filesystem::path& pageImagePath,
    const int& renderingWidth,
    const int& renderingHeight,
    const int& channels,
    const int& dpi,
    unsigned char* buff
    ) {
#ifdef _WIN32
    auto pageImagePathWcharString = SpecificOS::convertUTF8ToWchar(pageImagePath.string());
    FILE* file = _wfopen(pageImagePathWcharString.c_str(), L"wb");
#else
    FILE* file = fopen(pageImagePath.string().c_str(), "wb");
#endif

    png_structp png = png_create_write_struct(PNG_LIBPNG_VER_STRING, nullptr, nullptr, nullptr);
    png_infop info = png_create_info_struct(png);

    png_init_io(png, file);

    int colorType = PNG_COLOR_TYPE_RGB;
    if (channels == 1) {
        colorType = PNG_COLOR_TYPE_GRAY;
    } else if (channels == 3) {
        colorType = PNG_COLOR_TYPE_RGB;
    } else if (channels == 4) {
        colorType = PNG_COLOR_TYPE_RGBA;
    }

    png_set_IHDR(png, info, renderingWidth, renderingHeight, 8, colorType,
        PNG_INTERLACE_NONE, PNG_COMPRESSION_TYPE_DEFAULT,
        PNG_FILTER_TYPE_DEFAULT);

    auto pixelsPerMeter = static_cast<png_uint_32>(dpi / 0.0254);
    png_set_pHYs(png, info, pixelsPerMeter, pixelsPerMeter, PNG_RESOLUTION_METER);

    png_write_info(png, info);

    auto rowPointers = new png_bytep[renderingHeight];
    for (int y = 0; y < renderingHeight; ++y) {
        rowPointers[y] = buff + y * renderingWidth * channels;
    }

    png_write_image(png, rowPointers);
    png_write_end(png, info);

    delete[] rowPointers;
    png_destroy_write_struct(&png, &info);
    fclose(file);

    if (std::filesystem::exists(pageImagePath)) {
        return true;
    }

    return false;
}

bool SaveImage::savePDFToImage(
    const std::filesystem::path& pdfPath, const int& pageActualIndex, const std::filesystem::path& pageImagePath,
    const int& renderingWidthPixels, const int& renderingHeightPixels, const int& imageQuality, const int& dpi
    ) {
    auto [bufferBGRA, stride] = ReadPDF::getPDFPageBuffer(
        pdfPath, pageActualIndex, renderingWidthPixels, renderingHeightPixels
    );

    auto bufferRGB = bufferBGRAToRGB(
        bufferBGRA.data(), renderingWidthPixels, renderingHeightPixels, stride
    );

    std::string outputFormat = pageImagePath.extension().string().erase(0, 1);

    bool saveResult = false;
    if (outputFormat == "jpg") {
        saveResult = saveBufferToJPEG(
            pageImagePath,
            renderingWidthPixels,
            renderingHeightPixels,
            imageQuality,
            3,
            dpi,
            bufferRGB.data()
        );
    } else {
        saveResult = saveBufferToPNG(
            pageImagePath,
            renderingWidthPixels,
            renderingHeightPixels,
            3,
            dpi,
            bufferRGB.data()
        );
    }

    if (saveResult && std::filesystem::exists(pageImagePath)) {
        return true;
    }

    return false;
}