#include <iostream>
#include <ranges>
#include <filesystem>
#include <algorithm>

#ifdef _WIN32
#include <windows.h>

#include "specific_os.h"
#endif

#include "options.h"
#include "help.h"
#include "save_image.h"


int main(int argc, char *argv[]) {
    std::vector<std::string> argVector;

#ifdef _WIN32
    SetConsoleOutputCP(CP_UTF8);
    SetConsoleCP(CP_UTF8);

    argVector = SpecificOS::getArgvFromWchar();
#else
    for (int i = 0; i < argc; ++i) {
        std::string arg = argv[i];
        argVector.emplace_back(arg);
    }
#endif

    if (argc == 1) {
        Help::showHelp("--help");
        return 0;
    }

    for (const auto& arg : argVector) {
        if (arg == "-h" || arg == "-v" || arg == "--help") {
            Help::showHelp(arg);
            return 0;
        }
    }

    /**
     * std::filesystem::path pdfPath;
     * std::vector<std::filesystem::path> pageImagePaths;
     * std::vector<std::array<int, 2>> imageSizes;
     * std::vector<int> pageIndexes;
     * int imageQuality;
    */

    auto optionsResult = Options::getOptions(argc, argVector);

    if (optionsResult.has_value()) {
        auto [
            pdfPath,
            pageImagePaths,
            imageSizes,
            pageIndexes,
            imageQuality,
            dpi
        ] = optionsResult.value();

        std::cout << pdfPath << "\n\n";

        std::vector<bool> saveResults;

        for (int i = 0; i < pageIndexes.size(); ++i) {
            std::filesystem::path pageImagePath = pageImagePaths[i];
            auto [renderingWidthPixels, renderingHeightPixels] = imageSizes[i];

            bool saveResult = SaveImage::savePDFToImage(
                pdfPath,
                pageIndexes[i] - 1, pageImagePath,
                renderingWidthPixels, renderingHeightPixels,
                imageQuality,
                dpi
                );

            if (saveResult) {
                std::cout << pageIndexes[i] << " "
                << pageImagePath << " " << renderingWidthPixels << "x" << renderingHeightPixels
                << "\n\n";
            }

            saveResults.emplace_back(saveResult);
        }

        if (pageIndexes.size() == saveResults.size() && !std::ranges::contains(saveResults, false)) {
            return 0;
        }
    }

    std::cout << optionsResult.error() << "\n\n";

    return 1;
}
