#include "options.h"

#include <iostream>

#include "file_operations.h"
#include "format_handling.h"


std::expected<std::array<std::string, 4>, std::string>
Options::getOptions(int argc, const std::vector<std::string>& argVector) {
    // All available options.
    std::vector<std::string> options = {
        "-q",   // The query request
        "-e",   // The edit request
        "-c",   // The condition
    };

    std::string requestType = "query";
    std::string requestString;

    std::string jsonText;
    std::string queryString;
    std::string editString;
    std::string conditions;

    // argv[0] is the program's filename.
    int i = 1;
    while (i < argc) {
        const std::string& arg = argVector[i];
        if (arg == "-q" && i + 1 < argc) {
            queryString = argVector[i + 1];
            i = i + 2;
        } else if (arg == "-e" && i + 1 < argc) {
            editString = argVector[i + 1];
            i = i + 2;
        } else if (arg == "-c" && i + 1 < argc) {
            conditions = argVector[i + 1];
            i = i + 2;
        } else {
            if (std::filesystem::exists(arg)) {
                std::filesystem::path filePath = arg;
                jsonText = FileOperations::loadJSONFile(filePath);
            }
            ++i;
        }
    }

    if (jsonText.empty()) {
        std::cout << "No valid JSON file is found.\n"
                     "You may paste the text of complete JSON-format data here.\n"
                     "Once you finish inputting the text,\n"
                     "please press the key Enter twice for confirming.\n";

        std::string line;
        while (std::getline(std::cin, line)) {
            jsonText += line;
            if (line.empty()) {
                break;
            }
        }
    }

    if (jsonText.empty()) {
        return std::unexpected("No JSON text is found.");
    }

    queryString = FormatHandling::removeTwoSideSpace(queryString);
    editString = FormatHandling::removeTwoSideSpace(editString);
    conditions = FormatHandling::removeTwoSideSpace(conditions);

    if (queryString.empty() && editString.empty()) {
        return std::unexpected("There is no request.");
    }

    if (!editString.empty() && conditions.empty()) {
        return std::unexpected("Without any specific condition, an edit cannot be performed.");
    }

    if (!editString.empty()) {
        requestType = "edit";
        requestString = editString;
    } else if (!queryString.empty()) {
        requestType = "query";
        requestString = queryString;
    }

    return std::array{jsonText, requestType, requestString, conditions};
}