#include "data_handling.h"

#include <algorithm>
#include <ranges>
#include <sstream>
#include <iostream>

#include "format_handling.h"


/* ******************* *
 * Definition of marks *
 * ******************* */

std::vector<std::string> DataHandling::marks = {"[", "]", "{", "}", ","};

/* **************************************** *
 * Part I: Generate a flattening dictionary *
 * **************************************** */

/* For the first step, the JSON text is split into a vector by using " as a separator.
 * After the step, square brackets, braces and commas that perform functions in JSON
 * are exposed in the beginning or the end of the elements of the vector.
 * For the second step, square brackets and braces are stripped step by step.
 * Each square bracket or brace becomes a separate element of the vector.
 * Commas that perform a function, as well as spaces used for keeping the layout,
 * are all removed.
 * After the above process, the vector dict shows all keys and values with clear and
 * simple hierarchy.
 * Each square bracket or brace that performs a function becomes a separate element of
 * the vector, making the future query simple and easy to handle.
 */
std::string DataHandling::loadJSONText(const std::string& jsonTextRaw) {
    std::istringstream text(jsonTextRaw);
    std::string line;
    std::string jsonTextCombined;
    while (getline(text, line)) {
        jsonTextCombined += line;
    }

    return jsonTextCombined;
};

std::vector<std::string> DataHandling::splitJSON(const std::string& text) {
    auto splitJSONVector = text
    | std::views::split('"')
    | std::ranges::to<std::vector<std::string>>();

    return splitJSONVector;
}

std::vector<std::string> DataHandling::splitNest(const std::string& text) {
    std::vector<std::string> nest;

    for (auto& character : text) {
        std::string currentCharacter;
        currentCharacter += character;
        if (std::ranges::contains(marks, currentCharacter)) {
            nest.emplace_back(currentCharacter);
        } else if (currentCharacter != " ") {
            if (nest.empty() || std::ranges::contains(marks, nest.back())) {
                nest.emplace_back(currentCharacter);
            } else {
                nest.back() += currentCharacter;
            }
        }
    }

    return nest;
}

/* The function is designed for generating a flattening dictionary.
 * The structure of an element in the vector dict is like {key, {value's type, value}}.
 * The mark {, }, [ or ] is also stored as a key, while it's value is empty.
 */
std::vector<std::pair<std::string, std::pair<std::string, std::string>>>
DataHandling::generateDict(const std::string& jsonText) {
    std::vector<
        std::pair<
            std::string,
            std::pair<std::string, std::string>
        >
    > dict;

    /* A JSON text may contain the character '\n' that is not escaped, so the JSON text
     * should be combined into one line for making further processing easier.
     * When a JSON text is loaded from a file through loadJSONFile, it has been combined
     * into a line.
     * However, when a JSON text is from any other source, it may contains \n that is not
     * escaped, and should be processed by loadJSONText for combining into one line.
     * For simply processing, no matter which source the JSON text is from, it is always
     * processed using loadJSONText.
     * The characters "\n" which have been escaped are not handled at this step.
     */
    auto jsonTextCombined = loadJSONText(jsonText);

    /* As JSON uses double quotation marks to enclose key names and string values,
     * double quotation marks inside a string value are generally written as \" for escaping.
     * The library splits the JSON text by using " as a separator, so before splitting
     * the escaped double quotation mark \" is replaced with the mark's Unicode encoding \u0022.
     */
    auto escapedJSONText = FormatHandling::escapeDoubleQuotationMarks(jsonTextCombined);

    escapedJSONText = FormatHandling::removeTwoSideSpace(escapedJSONText);

    auto splitJSONVector = splitJSON(escapedJSONText);

    int j = 0;
    while (j < splitJSONVector.size()) {
        /* As the common structure of JSON is like "key: value",
         * three variables cleanI, cleanII and cleanIII are used for the analysis of the structure.
         */
        auto cleanI = FormatHandling::removeTwoSideSpace(splitJSONVector[j]);

        std::string cleanII;
        if (j + 1 < splitJSONVector.size()) {
            cleanII = FormatHandling::removeTwoSideSpace(splitJSONVector[j + 1]);
        }

        std::string cleanIII;
        if (j + 2 < splitJSONVector.size()) {
            cleanIII = FormatHandling::removeTwoSideSpace(splitJSONVector[j + 2]);
        }

        /* The first judgement is for recognizing a value which type is a string.
         *
         * In JSON, a key is always enclosed in double quotation marks, while
         * a value which type is a string is also enclosed in double quotation marks.
         * After using the double quotation mark " as the separator to split a JSON text,
         * keys are clearly separated, and values which type is a string are also clearly
         * separated. An empty-string value appears as an empty string in the vector of
         * split JSON.
         *
         * A colon implies the positions of a key and a value. Therefore, when a value's
         * type is a string, the key, the colon and the value are three separate elements
         * in the vector of split JSON.
         *
         * As the flattening dictionary stores all values as strings, for avoiding confusion,
         * when a value's type is a string, its type in the vector dict is written as "string".
         * Otherwise, the type is left empty.
         */
        if (!cleanI.empty() && cleanII == ":") {
            std::pair<
                std::string, std::pair<std::string, std::string>
            > pair = {cleanI, {"string", cleanIII}};

            dict.emplace_back(pair);

            j += 3;
        }

        /* The second judgement is for recognizing a value which type is not a string.
         * In JSON, when a value's type is not a string, it is not enclosed in
         * double quotation marks, so, after splitting JSON, the colon and the value
         * are not separated into two elements.
         * And the element may also contain braces or square brackets.
         * The function splitNest is designed for separating marks and any possible value.
         * As a layout may add spaces, the spaces between marks should be removed.
         * {, }, [ or ] is always in a separated element, and is regarded as the key of a pair.
         */
        else if (!cleanI.empty() && cleanII[0] == ':') {
            /* Remove the colon, which is the first character of the string. */
            std::string s = cleanII.erase(0, 1);

            /* If the last character of the string is a comma, remove it. */
            if (s[s.size() - 1] == ',') {
                s = s.erase(s.size() - 1, 1);
            }

            s = FormatHandling::removeTwoSideSpace(s);

            if (!s.empty()) {
                std::string lastChar;
                lastChar += s[s.size() - 1];

                /* The characters after a colon, ending with a comma or without a comma,
                 * may be a simple value, an object, or an array.
                 * A colon separates a key and a value, while a comma separates different pairs.
                 * After removing the colon and the possibly-existing comma,
                 * the last character of the remaining string can be the part of a simple value,
                 * the right brace }, or the right square bracket ].
                 * It can also be the left brace { or the left square bracket [
                 * if the end of the string looks like "}, {" or "], [".
                 * The following process distinguishes whether the last character of the string is
                 * a mark or not.
                 */
                if (std::ranges::contains(marks, lastChar)) {
                    /* The function splitNest splits the string between the colon and
                     * the possibly-existing comma into multiple elements of a vector.
                     */
                    auto nest = splitNest(s);

                    if (!nest.empty()) {
                        std::pair<std::string, std::pair<std::string, std::string>> pairKey;

                        /* If the first element is not a mark, it is a simple value.
                         * It may be a numerical value, or a boolean value or null.
                         * At this time it is not necessary to recognize what its type is,
                         * as the purpose of the library is for making a query but not for fully parsing.
                         * So here the type is just left empty.
                         */
                        if (!std::ranges::contains(marks, nest[0])) {
                            pairKey = {cleanI, {"", nest[0]}};
                            nest.erase(nest.begin());
                        }

                        /* If the first element is not a simple value, it should be the beginning of
                         * an array or an object, or a simple array like [1, 2, 3].
                         * In this case, cleanI is added to the vector dict as a key at the first.
                         */
                        else {
                            pairKey = {cleanI, {"", ""}};
                        }

                        dict.emplace_back(pairKey);

                        /* 1. If the first element is a simple value, the remaining elements should be
                         *    a mark like ] or }, or the combination of marks such as "}, {", "], {".
                         *    In this case, they are separately added to the vector dict as keys.
                         *
                         * 2. If the first element is not a simple value, the remaining elements
                         *    may be either of the following kinds:
                         * (1) The beginning mark { of an object, the beginning mark [ of an array,
                         *     or the combination of marks like [{. The marks are separately added
                         *     to the vector dict as keys.
                         * (2) An array with simple values which type is not a string,
                         *     such as [1, 2, 3] or [true, false, true]. The values are separately
                         *     added to the vector dict as the value of an element, while the key
                         *     of the element is left empty.
                         */
                        int k = 0;
                        while (k < nest.size()) {
                            if (nest[k] != ",") {
                                std::pair<
                                    std::string, std::pair<std::string, std::string>
                                > pair;

                                if (!std::ranges::contains(marks, nest[k])) {
                                    pair = {"", {"", nest[k]}};
                                } else {
                                    pair = {nest[k], {"", ""}};
                                }

                                dict.emplace_back(pair);
                            }

                            ++k;
                        }
                    }
                } else {
                    /* When the last character of the string is not a mark, it is a simple value.
                     */
                    std::pair<
                        std::string, std::pair<std::string, std::string>
                    > pair = {cleanI, {"", s}};

                    dict.emplace_back(pair);
                }
            }

            j += 2;
        }

        /* In this case, the remaining elements of the vector from split JSON
         * should be the marks which are located behind a string-type value in JSON.
         * For example, it may be a comma, the right brace, the right square bracket
         * or mixed marks like "}, {" or "]}, {".
         *
         * Moreover, if an array in JSON contains multiple string-type elements
         * like ["a", "b", "c"], it is recognized here, because the double quotation
         * mark makes the array be separated into multiple elements.
         *
         * If an array in JSON contains multiple non-string elements like [1, 2, 3] or
         * [true, false, true], it is recognized at the above step because it is generally
         * in a string like ": [1, 2, 3]" or ": [true, false, true]" after splitting JSON.
         *
         * If an array contains multi-type values like ["a", 1, null], the values are
         * processed here as the values are separated from ": [".
         */
        else {
            /* If the element is an empty string, it should be an empty-string value
             * in an array like ["a", "", "c"].
             * The empty-string value in the structure "key: value" has been processed in
             * the condition (!cleanI.empty() && cleanII == ":").
             */
            if (cleanI.empty()) {
                std::pair<
                    std::string, std::pair<std::string, std::string>
                > pair = {"", {"", cleanI}};

                dict.emplace_back(pair);
            } else {
                if (cleanI[cleanI.size() - 1] == ',') {
                    cleanI.erase(cleanI.size() - 1, 1);
                }

                cleanI = FormatHandling::removeTwoSideSpace(cleanI);

                if (!cleanI.empty()) {
                    std::string lastChar;
                    lastChar += cleanI[cleanI.size() - 1];

                    /* If a string is like "true]", its type is not a string in JSON.
                     * So, it is clear that if the end of a string is not a mark,
                     * it's type should be a string in JSON.
                     *
                     * In an array like ["a", "b", "c"], there is no key,
                     * so the values in the array are regarded as the value of an element
                     * in the vector dict, and the key of the element is left empty.
                     * If the array is like ["a", "b", ""], the last value is also an
                     * empty-string value of the element in the vector dict.
                     *
                     * Why leave the key of the element empty for the simple values of an array?
                     * It can be clearly differentiated from an object's key that has an empty value
                     * which type is a string.
                     */
                    if (!std::ranges::contains(marks, lastChar)) {
                        std::pair<
                            std::string, std::pair<std::string, std::string>
                        > pair = {"", {"string", cleanI}};

                        dict.emplace_back(pair);
                    } else {
                        auto nest = splitNest(cleanI);

                        /* No matter how many types the values of an array, any string starting with
                         * a colon is not processed here, but has been processed in the above steps.
                         * Therefore, here the elements of the vector generated by splitNest are
                         * just added to the vector dict. If the element is not a mark, it should be
                         * a value of an array.
                         */
                        for (auto& part : nest) {
                            if (part != ",") {
                                std::pair<
                                    std::string, std::pair<std::string, std::string>
                                > pair;
                                if (std::ranges::contains(marks, part)) {
                                    pair = {part, {"", ""}};
                                } else {
                                    pair = {"", {"", part}};
                                }

                                dict.emplace_back(pair);
                            }
                        }
                    }
                }
            }

            ++j;
        }
    }

    return dict;
}

/* ********************************************************* *
 * Part II: Data query and edit in the flattening dictionary *
 * ********************************************************* */

std::pair<std::string, std::string>
DataHandling::getKeysAndValue(std::string stringRaw) {
    std::string keys;
    std::string value;

    for (auto& character : stringRaw) {
        if (character == '=') {
            break;
        }

        keys += character;
    }

    if (keys.size() < stringRaw.size()) {
        value = stringRaw.erase(0, keys.size() + 1);
    }

    keys = FormatHandling::removeTwoSideSpace(keys);
    value = FormatHandling::removeTwoSideSpace(value);

    return {keys, value};
}

std::tuple<
    std::vector<std::pair<std::vector<std::string>, std::string>>,
    std::vector<std::pair<std::vector<std::string>, std::string>>
>
DataHandling::recognizeConditions(
    const std::string& requestConditions, const std::string& queryKeys
    ) {
    std::vector<std::pair<std::vector<std::string>, std::string>> sameHierarchyConditionVector;
    std::vector<std::pair<std::vector<std::string>, std::string>> conditionVector;

    auto queryVector = queryKeys
    | std::views::split('.')
    | std::ranges::to<std::vector<std::string>>();

    auto conditions = requestConditions
    | std::views::split(',')
    | std::ranges::to<std::vector<std::string>>();

    for (auto& condition : conditions) {
        auto [conditionKeys, conditionValue] = getKeysAndValue(condition);

        if (!conditionKeys.empty() && !conditionValue.empty()) {
            auto conditionKeyVector = conditionKeys
            | std::views::split('.')
            | std::ranges::to<std::vector<std::string>>();

            bool isSameHierarchy = true;

            if (queryVector.size() == conditionKeyVector.size()) {
                for (int k = 0; k < queryVector.size(); ++k) {
                    if (k < queryVector.size() - 1) {
                        if (queryVector[k] != conditionKeyVector[k]) {
                            isSameHierarchy = false;
                        }
                    }
                }
            } else {
                isSameHierarchy = false;
            }

            std::pair<std::string, std::string> conditionStringPair = {
                conditionKeys, conditionValue
            };

            std::pair<std::vector<std::string>, std::string> conditionPair = {
                conditionKeyVector, conditionValue
            };

            if (isSameHierarchy) {
                sameHierarchyConditionVector.emplace_back(conditionPair);
            } else {
                conditionVector.emplace_back(conditionPair);
            }
        }
    }

    return std::make_tuple(sameHierarchyConditionVector, conditionVector);
}

/* The function is for updating the key path, which means the hierarchy keys,
 * based on the location where an element in the vector dict is being visited.
 */
std::vector<std::string> DataHandling::updateHierarchy(
    std::vector<std::string> hierarchyVector,
    const std::string& key,
    int braceIndex
    ) {
    if (!std::ranges::contains(marks, key) && !key.empty()) {
        if (hierarchyVector.size() > braceIndex + 1) {
            hierarchyVector.pop_back();
        }

        if (hierarchyVector.size() > braceIndex) {
            hierarchyVector[braceIndex] = key;
        }

        if (hierarchyVector.size() == braceIndex) {
            hierarchyVector.emplace_back(key);
        }
    }

    return hierarchyVector;
}

bool DataHandling::compareVectors(
    const std::vector<std::string>& vector1,
    const std::vector<std::string>& vector2
    ) {
    bool isSame = true;

    if (vector1.size() == vector2.size()) {
        for (int i = 0; i < vector1.size(); ++i) {
            if (vector1[i] != vector2[i]) {
                isSame = false;
            }
        }
    } else {
        isSame = false;
    }

    return isSame;
}

std::pair<std::array<int, 3>, std::pair<std::string, std::string>>
DataHandling::getArrayFromDict(
    const std::vector<std::pair<std::string, std::pair<std::string, std::string>>>& dict,
    int objectIndexInArray,
    int arrayBeginningIndexInDict
    ) {
    std::vector<std::pair<std::string, std::pair<std::string, std::string>>> arrayElementVector;

    int squareBracketIndex = -1;

    for (int i = arrayBeginningIndexInDict; i < dict.size(); ++i) {
        arrayElementVector.emplace_back(dict[i]);

        if (dict[i].first == "[") {
            ++squareBracketIndex;
        } else if (dict[i].first == "]") {
            --squareBracketIndex;
        }

        if (squareBracketIndex == -1) {
            break;
        }
    }

    std::string arrayData = generateJSONFromDict(arrayElementVector);

    int elementCount = static_cast<int>(arrayElementVector.size());

    return {
            {objectIndexInArray, arrayBeginningIndexInDict, elementCount},
            {"array", arrayData}
    };
}

std::pair<std::array<int, 3>, std::pair<std::string, std::string>>
DataHandling::getObjectFromDict(
    const std::vector<std::pair<std::string, std::pair<std::string, std::string>>>& dict,
    int objectIndexInArray,
    int objectBeginningIndexInDict
    ) {
    std::vector<std::pair<std::string, std::pair<std::string, std::string>>> objectElementVector;

    int braceIndex = -1;

    for (int i = objectBeginningIndexInDict; i < dict.size(); ++i) {
        objectElementVector.emplace_back(dict[i]);

        if (dict[i].first == "{") {
            ++braceIndex;
        } else if (dict[i].first == "}") {
            --braceIndex;
        }

        if (braceIndex == -1) {
            break;
        }
    }

    std::string objectData = generateJSONFromDict(objectElementVector);

    int elementCount = static_cast<int>(objectElementVector.size());

    return {
        {objectIndexInArray, objectBeginningIndexInDict, elementCount},
        {"object", objectData}
    };
}

/* Generate JSON from a flattening dictionary. */
std::string DataHandling::generateJSONFromDict(
    const std::vector<std::pair<std::string, std::pair<std::string, std::string>>>& dict
    ) {
    std::string jsonText;

    for (int i = 0; i < dict.size(); ++i) {
        bool canAddComma = true;

        /* If the key is not empty, it may imply the structure "key: value",
         * or just a mark.
         */
        if (!dict[i].first.empty()) {
            /* If the type is "string", no matter what the value is,
             * it is the structure "key: value".
             * Double quotation marks should enclose the key and the value.
             */
            if (dict[i].second.first == "string") {
                jsonText += "\"";
                jsonText += dict[i].first;
                jsonText += "\"";
                jsonText += ": ";
                jsonText += "\"";
                jsonText += dict[i].second.second;
                jsonText += "\"";
            }

            /* If the type is not "string" and the value is not empty,
             * it is also the structure "key: value",
             * but the value's type is not a string in JSON.
             */
            else if (!dict[i].second.second.empty())  {
                jsonText += "\"";
                jsonText += dict[i].first;
                jsonText += "\"";
                jsonText += ": ";
                jsonText += dict[i].second.second;
            }

            /* Now the value is empty and its type is not "string".
             * If the key is not a mark, it is a key in JSON and its value
             * should be an object or an array.
             */
            else if (!std::ranges::contains(marks, dict[i].first)) {
                jsonText += "\"";
                jsonText += dict[i].first;
                jsonText += "\"";
                jsonText += ": ";

                canAddComma = false;
            }

            /* Now only the key that is a mark is left.
             */
            else {
                jsonText += dict[i].first;

                if (dict[i].first == "[" || dict[i].first == "{") {
                    canAddComma = false;
                }
            }
        }

        /* If the key is empty, the value should be in an array.
         */
        else {
            /* If the value's type is "string",
             * the value's type is a string in JSON,
             * so the value should be enclosed in double quotation marks.
             */
            if (dict[i].second.first == "string") {
                jsonText += "\"";
                jsonText += dict[i].second.second;
                jsonText += "\"";
            }

            /* If the value's type is not "string",
             * the value's type is not a string in JSON.
             */
            else {
                jsonText += dict[i].second.second;
            }
        }

        /* Add a comma when necessary. */
        if (canAddComma && !jsonText.empty() && i < dict.size() - 1) {
            if (dict[i + 1].first != "]" && dict[i + 1].first != "}") {
                jsonText += ", ";
            }
        }
    }

    return jsonText;
}

/* The function is designed for making a query through a one-time visit to
 * all elements of the vector dict.
 * Currently, it is able to process a query to a simple JSON text that complies
 * with JSON standards.
 * For multi-layer complicated JSON, there is a lot of work to do.
 */
std::tuple<bool, std::pair<std::string, std::string>, std::string>
DataHandling::handleRequest(
    const std::string& requestType,
    const std::string& jsonData,
    const std::string& queryKeys,
    const std::string& editValue,
    const std::string& requestConditions
    ) {
    bool isSuccessful = false;
    std::pair<std::string, std::string> queryResult;
    std::string editedJSONData;

    auto dict = generateDict(jsonData);

    // Test Start
    // Show all elements of dict
    //for (auto& [key, value] : dict) {
    //    if (value.first == "string") {
    //        std::cout << key << ": \"" << value.second << "\"" << std::endl;
    //    } else if (!value.second.empty()) {
    //        std::cout << key << ": " << value.second << std::endl;
    //    } else {
    //        std::cout << key << std::endl;
    //    }
    //}
    // Show JSON generated from dict
    //std::cout << generateJSONFromDict(dict) << std::endl;
    // Test End

    auto queryVector = queryKeys
    | std::views::split('.')
    | std::ranges::to<std::vector<std::string>>();

    /* Same hierarchy conditions means that their final key is in the same object of the query's final key.
     * For example, phone_numbers.type and phone_numbers.number are in the same hierarchy,
     * so phone_numbers.type=home can be the same hierarchy condition of the query phone_numbers.number,
     * but uer_id=12345 is not a same hierarchy condition.
     *
     * A same hierarchy condition is very important to locate a query if there are multiple pairs with same keys
     * in an object.
     */
    const auto [sameHierarchyConditionVector, conditionVector] = recognizeConditions(requestConditions, queryKeys);

    /* The default value is set to -1, because the index of the first element in an array is 0,
     * and -1 can clearly distinguish whether an array is being visited.
     * objectIndexInArray is the index of an object in the array,
     * foundObjectIndexInArray is the index of the object that contains the query's result.
     */
    int squareBracketIndex = -1;
    int braceIndex = -1;
    int objectIndexInArray = -1;
    int foundObjectIndexInArray = -1;

    /* How many the conditions are, how many default values are set in the vectors.
     * if the string of hierarchy keys is computer.hardware.cpu.model, after the keys are separated
     * into a vector, the index of computer is 0, hardware's is 1, cpu's is 2, and model's is 3.
     */
    std::vector<std::string> emptyVector;

    int sameHierarchyConditionCount = static_cast<int>(sameHierarchyConditionVector.size());
    std::vector<int> sameHierarchyConditionIndexVector(sameHierarchyConditionCount, 0);
    std::vector<bool> sameHierarchyConditionMatchVector(sameHierarchyConditionCount, false);
    std::vector<std::vector<std::string>> sameHierarchyConditionHierarchy(sameHierarchyConditionCount, emptyVector);

    /* The bool variable is to recognize if the judgement for same hierarchy conditions can be ended.
     * Only when all same hierarchy conditions are satisfied, the variable's value becomes true.
     */
    bool isSameHierarchyMatching = false;

    int conditionCount = static_cast<int>(conditionVector.size());
    std::vector<int> conditionIndexVector(conditionCount, 0);
    std::vector<bool> conditionMatchVector(conditionCount, false);
    std::vector<std::vector<std::string>> conditionHierarchy(conditionCount, emptyVector);

    std::vector<std::string> queryHierarchy;

    /* In an element of the vector possibleQueryResult:
     *
     * (1) The first int is objectIndexInArray, which means the index of an object in an array.
     *     It is a relative index.
     *     The index of the object that contains the query's exact result is foundObjectIndexInArray,
     *     decided by same hierarchy conditions.
     *     When same hierarchy conditions are given, it is helpful to locate the query result
     *     for comparing foundObjectIndexInArray with objectIndexInArray.
     *     If no same hierarchy condition is given, foundObjectIndexInArray keeps -1, and
     *     objectIndexInArray is not used for comparison with it.
     *
     * (2) The second int can be the index of an element in the vector dict, if the query result
     *     is just a string or another kind of simple value;
     *     when the query result is an object or an array, the result contains the contents of
     *     multiple elements in the vector dict, so the second int should be the beginning index
     *     of elements in the vector dict.
     *     It is an absolute index.
     *
     * (3) The third int is the number of elements in the vector dict as the value of the query
     *     result. If the query result is a string or another kind of simple value, the number is 1.
     *     But if the query result is an object or an array, as an object or an array are separated
     *     into multiple elements in the vector dict, the number should be the count of all elements
     *     from the object or the array.
     *
     * The vector possibleQueryResult may contain multiple elements, if in the array there are
     * multiple objects with same keys. As all elements in the vector dict are visited in sequence,
     * the key path of a query, which means the query's hierarchy keys, may be found before or after
     * foundObjectIndexInArray gets an exact value, so the information of all possible objects that
     * contain the same key path should be added to possibleQueryResult, and after visiting all
     * elements of the vector dict, the exact query result can be read from possibleQueryResult
     * using foundObjectIndexInArray.
     */
    std::vector<std::pair<std::array<int, 3>, std::pair<std::string, std::string>>> possibleQueryResult;

    /* The default value is set to 0, because the index of the first element in a vector is 0,
     * and it is visited firstly.
     */
    int i = 0;
    int queryVectorIndex = 0;

    /* objectIndexInArray is to recognize the relative index of an object in an array,
     * and the bool variable isInArray is used for recognizing if an array is being
     * visited, as an object may be not in an array. For example, the top-layer objects
     * are not in an array.
     */
    bool isInArray = false;

    while (i < dict.size()) {
        if (dict[i].first == "{") {
            ++braceIndex;
            if (isInArray) {
                ++objectIndexInArray;
            }
        }

        else if (dict[i].first == "}") {
            --braceIndex;
        }

        else if (dict[i].first == "[") {
            ++squareBracketIndex;
            objectIndexInArray = -1;
            isInArray = true;
        }

        else if (dict[i].first == "]") {
            --squareBracketIndex;
            objectIndexInArray = -1;
            isInArray = false;
        }

        for (int j = 0; j < conditionVector.size(); ++j) {
            /* If a condition is satisfied, it does not need to compare the condition's
             * key path anymore.
             */
            if (conditionMatchVector[j]) {
                continue;
            }

            auto [conditionKeys, conditionValue] = conditionVector[j];

            conditionHierarchy[j] = updateHierarchy(conditionHierarchy[j], dict[i].first, braceIndex);

            if (braceIndex == conditionIndexVector[j]
                && dict[i].first == conditionKeys[conditionIndexVector[j]]
                ) {
                if (conditionIndexVector[j] + 1 < conditionKeys.size()) {
                    ++conditionIndexVector[j];
                }

                if (compareVectors(conditionHierarchy[j], conditionKeys)
                    && dict[i].second.second == conditionValue
                ) {
                    conditionMatchVector[j] = true;
                }
            }
        }

        if (!sameHierarchyConditionVector.empty()) {
            for (int k = 0; k < sameHierarchyConditionVector.size(); ++k) {
                /* If isSameHierarchyMatching is true, it means all same hierarchy conditions are
                 * satisfied, so the judgement for same hierarchy conditions can be ended.
                 */
                if (isSameHierarchyMatching) {
                    break;
                }

                auto [conditionKeys, conditionValue] = sameHierarchyConditionVector[k];

                sameHierarchyConditionHierarchy[k] = updateHierarchy(
                    sameHierarchyConditionHierarchy[k], dict[i].first, braceIndex);

                /* In an array, when multiple objects have same keys and multiple same hierarchy conditions
                 * are given, the matching state of each same hierarchy condition should be reset to false
                 * when finishing the visit to an object.
                 * The change of braceIndex indicates whether an object is being visited.
                 */
                if (braceIndex != sameHierarchyConditionIndexVector[k]) {
                    sameHierarchyConditionMatchVector.assign(sameHierarchyConditionCount, false);
                } else if (dict[i].first == conditionKeys[sameHierarchyConditionIndexVector[k]]) {
                    if (sameHierarchyConditionIndexVector[k] + 1 < conditionKeys.size()) {
                        ++sameHierarchyConditionIndexVector[k];
                    }

                    if (compareVectors(sameHierarchyConditionHierarchy[k], conditionKeys)
                        && dict[i].second.second == conditionValue
                    ) {
                        sameHierarchyConditionMatchVector[k] = true;

                        /* If multiple same hierarchy conditions are given, only when
                         * all the conditions are satisfied, foundObjectIndexInArray
                         * then is confirmed.
                         */
                        int satisfiedCount = 0;
                        for (const auto& conditionMatch : sameHierarchyConditionMatchVector) {
                            if (conditionMatch) {
                                ++satisfiedCount;
                            }
                        }

                        if (satisfiedCount == sameHierarchyConditionVector.size()) {
                            isSameHierarchyMatching = true;
                            foundObjectIndexInArray = objectIndexInArray;
                        }
                    }
                }
            }
        }

        queryHierarchy = updateHierarchy(queryHierarchy, dict[i].first, braceIndex);

        if (braceIndex == queryVectorIndex && dict[i].first == queryVector[queryVectorIndex]) {
            if (queryVectorIndex + 1 < queryVector.size()) {
                ++queryVectorIndex;
            }

            if (compareVectors(queryHierarchy, queryVector)) {
                /* If the value's type is "string", the value can be empty;
                 * but if the type is not "string" and the value is empty,
                 * this means in JSON the key's value is an object or an array.
                 */
                if (dict[i].second.first == "string" || !dict[i].second.second.empty()) {
                    std::pair<std::array<int, 3>, std::pair<std::string, std::string>>
                    possiblePair = {
                        {objectIndexInArray, i, 1},
                        dict[i].second
                    };
                    possibleQueryResult.emplace_back(possiblePair);
                } else {
                    /* When the key's value is an array or an object. */
                    if (dict[i + 1].first == "[") {
                        auto possiblePair = getArrayFromDict(dict, objectIndexInArray, i + 1);
                        possibleQueryResult.emplace_back(possiblePair);
                    } else if (dict[i + 1].first == "{") {
                        auto possiblePair = getObjectFromDict(dict, objectIndexInArray, i + 1);
                        possibleQueryResult.emplace_back(possiblePair);
                    }
                }
            }
        }

        ++i;
    }

    bool isMatching = true;

    for (const auto& conditionMatch : conditionMatchVector) {
        if (!conditionMatch) {
            isMatching = false;
        }
    }

    /* The default value of isSameHierarchyMatching is false, and the change of the
     * value is decided by the judgement for same hierarchy conditions.
     * If one or multiple same hierarchy conditions are given, isSameHierarchyMatching
     * still keeps false, this means that at lease one of the same hierarchy conditions
     * is not satisfied.
     */
    if (!isSameHierarchyMatching && !sameHierarchyConditionVector.empty()) {
        isMatching = false;
    }

    /* When foundObjectIndexInArray is -1, it may be caused by one of the three reasons:
     *
     * (1) No same hierarchy condition is given.
     *     In this case, the value of isMatching is decided by the judgement for
     *     conditionMatchVector.
     *
     * (2) One or multiple same hierarchy conditions are given, and at least one of
     *     the same hierarchy conditions is not satisfied.
     *     In this case, the value of the variable isSameHierarchyMatching is false.
     *
     * (3) One or multiple same hierarchy conditions are given, though all of them
     *     are satisfied, but the found key path is not in an array, which means
     *     objectIndexInArray keeps -1 and foundObjectIndexInArray accordingly keeps -1.
     *     In this case, the value of isSameHierarchyMatching is true.
     *
     * Since isMatching is true, when foundObjectIndexInArray is -1, it just means (1) or (3).
     * Therefore, if the query's key path is in an array, the value of the found key in the
     * first object of the array is regarded as the query's final result.
     *
     * When foundObjectIndexInArray is not -1, the exact result of an query can be found
     * by finding the same objectIndexInArray in possibleQueryResult.
     *
     * If the value of the found key is an object or an array, the entire data of the object or
     * the array is the query's result.
     */
    if (isMatching) {
        if (foundObjectIndexInArray != -1) {
            for (auto& [possibleKey, possibleValue] : possibleQueryResult) {
                if (std::get<0>(possibleKey) == foundObjectIndexInArray) {
                    if (requestType == "query") {
                        queryResult = possibleValue;
                        isSuccessful = true;
                    }

                    else if (requestType == "edit") {
                        if (possibleValue.first != "object" && possibleValue.first != "array") {
                            dict[std::get<1>(possibleKey)].second.second = editValue;
                            isSuccessful = true;
                            editedJSONData = generateJSONFromDict(dict);
                        } else {
                            /* Firstly all elements related to the value of the found key are removed. */
                            dict.erase(
                                dict.begin() + std::get<1>(possibleKey),
                                dict.begin() + std::get<1>(possibleKey) + std::get<2>(possibleKey)
                            );

                            /* If the value of a key is an object or an array, the key's index
                             * is before the index of the beginning of the object or the array.
                             */
                            dict[std::get<1>(possibleKey) - 1].second.second = editValue;
                            isSuccessful = true;
                            editedJSONData = generateJSONFromDict(dict);
                        }

                    }
                }
            }
        } else if (!possibleQueryResult.empty()) {
            auto foundElement = possibleQueryResult[0];

            if (requestType == "query") {
                queryResult = foundElement.second;
                isSuccessful = true;
            }

            else if (requestType == "edit") {
                if (foundElement.second.first != "object" && foundElement.second.first != "array") {
                    dict[std::get<1>(foundElement.first)].second.second = editValue;
                    isSuccessful = true;
                    editedJSONData = generateJSONFromDict(dict);
                } else {
                    dict.erase(
                        dict.begin() + std::get<1>(foundElement.first),
                        dict.begin() + std::get<1>(foundElement.first) + std::get<2>(foundElement.first)
                    );

                    dict[std::get<1>(foundElement.first) - 1].second.second = editValue;
                    isSuccessful = true;
                    editedJSONData = generateJSONFromDict(dict);
                }
            }
        }
    }

    return std::make_tuple(isSuccessful, queryResult, editedJSONData);
}

std::tuple<bool, std::pair<std::string, std::string>, std::string> DataHandling::dataQuery(
        const std::string& jsonData,
        const std::string& queryString,
        const std::string& queryConditions
        ) {
    auto queryResult = handleRequest(
        "query", jsonData, queryString, "", queryConditions
    );

    return queryResult;
}

std::tuple<bool, std::pair<std::string, std::string>, std::string> DataHandling::dataEdit(
    const std::string& jsonData,
    const std::string& editString,
    const std::string& editConditions
    ) {
    std::tuple<bool, std::pair<std::string, std::string>, std::string> editResult;

    auto [queryKeys, editValue] = getKeysAndValue(editString);

    if (!queryKeys.empty() && !editValue.empty()) {
        editResult = handleRequest(
            "edit", jsonData, queryKeys, editValue, editConditions
        );
    }

    return editResult;
}
