#include "options.h"

#include <iostream>
#include <ranges>
#include <algorithm>
#include <random>


std::vector<std::filesystem::path> Options::getImagePaths(const std::string& str) {
    std::vector<std::filesystem::path> imagePaths;

    auto possiblePaths = str
    | std::views::split(',')
    | std::ranges::to<std::vector<std::string>>();

    for (auto& possiblePath : possiblePaths) {
        std::filesystem::path path(possiblePath);
        if (std::filesystem::exists(path)) {
            std::string fileExtension = path.extension().string();

            std::ranges::transform(
                fileExtension, fileExtension.begin(),
                [](unsigned char c) { return std::tolower(c); }
                );

            if (fileExtension == ".png" || fileExtension == ".jpeg" || fileExtension == ".jpg") {
                imagePaths.emplace_back(path);
            }
        }
    }

    return imagePaths;
}

std::array<int, 4> Options::getMargins(const std::string& str) {
    std::array<int, 4> margins{0, 0, 0, 0};

    auto possibleMargins = str
    | std::views::split(',')
    | std::ranges::to<std::vector<std::string>>();

    std::vector<int> possibleMarginVector;
    for (auto& possibleMargin : possibleMargins) {
        auto getDigits = getDigitsFromString(possibleMargin);
        if (getDigits.has_value() && getDigits.value() > 0) {
            possibleMarginVector.emplace_back(getDigits.value());
        } else {
            possibleMarginVector.emplace_back(0);
        }
    }

    if (possibleMarginVector.size() == 1) {
        margins = {
            possibleMarginVector[0], possibleMarginVector[0], possibleMarginVector[0], possibleMarginVector[0]
        };
    }

    else if (possibleMarginVector.size() == 2) {
        margins = {
            possibleMarginVector[0], possibleMarginVector[1], 0, 0
        };
    }

    else if (possibleMarginVector.size() == 3) {
        margins = {
            possibleMarginVector[0], possibleMarginVector[1], possibleMarginVector[2], 0
        };
    }

    else if (possibleMarginVector.size() >= 4) {
        margins = {
            possibleMarginVector[0], possibleMarginVector[1], possibleMarginVector[2], possibleMarginVector[3]
        };
    }

    return margins;
}

std::array<int, 2> Options::getSpacings(const std::string& str) {
    std::array<int, 2> spacings{0, 0};

    auto possibleSpacings = str
    | std::views::split(',')
    | std::ranges::to<std::vector<std::string>>();

    std::vector<int> possibleSpacingVector;
    for (auto& possibleSpacing : possibleSpacings) {
        auto getDigits = getDigitsFromString(possibleSpacing);
        if (getDigits.has_value() && getDigits.value() > 0) {
            possibleSpacingVector.emplace_back(getDigits.value());
        } else {
            possibleSpacingVector.emplace_back(0);
        }
    }

    if (possibleSpacingVector.size() == 1) {
        spacings = {possibleSpacingVector[0], possibleSpacingVector[0]};
    }

    else if (possibleSpacingVector.size() >= 2) {
        spacings = {possibleSpacingVector[0], possibleSpacingVector[1]};
    }

    return spacings;
}

std::expected<int, std::string> Options::getDigitsFromString(const std::string& str) {
    std::string digitString = str
    | std::views::filter([](unsigned char c) { return std::isdigit(c); })
    | std::ranges::to<std::string>();

    if (!digitString.empty()) {
        return std::stoi(digitString);
    }

    return std::unexpected("There is no digital.");
}

std::expected<
    std::tuple<
        std::vector<std::filesystem::path>,
        std::filesystem::path,
        std::array<int, 2>,
        int,
        int,
        int,
        std::array<int, 4>,
        std::array<int, 2>
    >,
    std::string
>
Options::getOptions(const int& argc, const std::vector<std::string>& argVector) {
    // All available options.
    std::vector<std::string> options = {
        "-i",  // The input image paths.
        "-o",  // The output pdf path.
        "-p",  // The size of paper in millimeters.
        "-w",  // The output width of images in millimeters.
        "-q",  // The output quality of images, an integer between 1 and 100.
        "-n",   // The number of images per page.
        "--set-margin",  // Top margin, bottom margin, left margin, right margin.
        "--set-spacing"  // Horizontal spacing of images, vertical spacing of images.
    };

    std::vector<std::filesystem::path> imagePaths;
    std::filesystem::path pdfPath;

    std::array<int, 2> paperSize = {0, 0};
    int outputWidth = 0;
    int outputQuality = 0;
    int imageNumberPerPage = 1;
    std::array<int, 4> margins{0, 0, 0, 0};
    std::array<int, 2> spacings{0, 0};

    std::vector<std::string> filtered_options;

    /* Distinguish all arguments.
     * The first element from argv is the program's name, which should be skipped.
     */
    for (int i = 1; i < argc; ++i) {
        std::string arg = argVector[i];

        /* Recognize options.
         * An option with its value can be written like "-d300" or "-d 300".
         */
        if (arg.size() >= 2 && std::ranges::contains(options, arg.substr(0, 2))) {
            filtered_options.emplace_back(arg.substr(0, 2));

            if (arg.size() > 2) {
                filtered_options.emplace_back(arg.substr(2));
            }
        }

        else if (arg.size() >= 12 && std::ranges::contains(options, arg.substr(0, 12))) {
            filtered_options.emplace_back(arg.substr(0, 12));

            if (arg.size() > 12) {
                filtered_options.emplace_back(arg.substr(12));
            }
        }

        else if (arg.size() >= 13 && std::ranges::contains(options, arg.substr(0, 13))) {
            filtered_options.emplace_back(arg.substr(0, 13));

            if (arg.size() > 13) {
                filtered_options.emplace_back(arg.substr(13));
            }
        }

        /* Recognize the value of each option. Because of spaces, the value of an option may
         * be separated to multiple arguments, and here the task is only to combine separate
         * arguments into one element as the possible value of an option.
         */
        else if (!filtered_options.empty()) {
            std::string& lastArg = filtered_options.back();

            if (std::ranges::contains(options, lastArg)) {
                filtered_options.emplace_back(arg);
            } else {
                std::vector<std::string> multiValueOptions = {"-i", "--set-margin", "--set-spacing"};
                /* The list of images can be written as "image1.jpg,image2.jpg,image3.jpg",
                 * "image1.jpg, image2.jpg, image3.jpg" or "image1.jpg image2.jpg image3.jpg".
                 */
                if (filtered_options.size() >= 2
                    && std::ranges::contains(multiValueOptions, filtered_options[filtered_options.size() - 2])
                    ) {
                    if (lastArg[lastArg.size() - 1] == ',') {
                        filtered_options.back() += arg;
                    } else {
                        filtered_options.back() += ',';
                        filtered_options.back() += arg;
                    }
                } else {
                    /* If the value of an option is written like "210 x 297",
                     * it can be correctly combined as "210x297".
                     */
                    filtered_options.back() += arg;
                }
            }
        }
    }

    // Remove invalid options.
    for (int i = 0; i < filtered_options.size(); ++i) {
        if (std::ranges::contains(options, filtered_options[i])
            && std::ranges::contains(options, filtered_options[i + 1])
            ) {
            filtered_options[i] = "";
        }
    }

    std::erase(filtered_options, "");

    // Recognize options.
    for (std::string& option: options) {
        auto viewResult = filtered_options
        | std::views::drop_while([option](auto& filteredOption){ return filteredOption != option; })
        | std::views::drop(1)
        | std::views::take(1);

        if (viewResult.empty()) {
            continue;
        }

        auto& optionValue = *viewResult.begin();

        if (option == "-i") {
            imagePaths = getImagePaths(optionValue);
        }

        else if (option == "-o") {
            pdfPath = std::filesystem::path(optionValue);
        }

        else if (option == "-p") {
            auto paperSizeStrings = optionValue
                | std::views::split('x')
                | std::ranges::to<std::vector<std::string>>();

            std::vector<int> paperSizeDigits;
            for (std::string& paperSizeString : paperSizeStrings) {
                auto getDigits = getDigitsFromString(paperSizeString);
                if (getDigits.has_value()) {
                    paperSizeDigits.emplace_back(getDigits.value());
                }
            }

            if (paperSizeDigits.size() == 2) {
                paperSize = std::array{paperSizeDigits[0], paperSizeDigits[1]};
            }
        }

        else if (option == "-w") {
            auto getDigits = getDigitsFromString(optionValue);
            if (getDigits.has_value()) {
                outputWidth = getDigits.value();
            }
        }

        else if (option == "-q") {
            auto getDigits = getDigitsFromString(optionValue);
            if (getDigits.has_value()) {
                if (getDigits.value() > 0 && getDigits.value() <= 100) {
                    outputQuality = getDigits.value();
                }
            }
        }

        else if (option == "-n") {
            auto getDigits = getDigitsFromString(optionValue);
            if (getDigits.has_value()) {
                if (getDigits.value() > 0) {
                    imageNumberPerPage = getDigits.value();
                }
            }
        }

        else if (option == "--set-margin") {
            margins = getMargins(optionValue);
        }

        else if (option == "--set-spacing") {
            spacings = getSpacings(optionValue);
        }

    }

    if (imagePaths.empty()) {
        return std::unexpected("No image path is given.");
    }

    if (pdfPath.empty()) {
        auto baseName = std::filesystem::path(imagePaths[0]).stem().string();

        auto generateRandomString =[]() {
            const std::string baseChars = "0123456789abcdefghijklmnopqrstuvwxz";

            std::random_device rd;
            std::mt19937 gen(rd());
            std::uniform_int_distribution<size_t> distrib(0, baseChars.size() - 1);

            std::string randomString;
            constexpr int randomStringSize = 8;
            for (int i = 0; i < randomStringSize; ++i) {
                randomString += baseChars[distrib(gen)];
            }

            return randomString;
        };

        pdfPath = std::filesystem::path(baseName + "_" + generateRandomString() + ".pdf");
    }

    return std::make_tuple(
        imagePaths,
        pdfPath,
        paperSize,
        outputWidth,
        outputQuality,
        imageNumberPerPage,
        margins,
        spacings
    );
}