#include "calculator.h"

#include <algorithm>
#include <cmath>
#include <iostream>


int Calculator::convertMMToPoints(const int& millimeters) {
    /* 1 inch = 25.4 mm, 1 point = 1/72 inch. */
    int points = static_cast<int>(std::round(static_cast<double>(millimeters) / 25.4 * 72.0));
    return points;
}

int Calculator::convertMMToPixels(const int& millimeters, const int& dpi) {
    int pixels = static_cast<int>(std::round(static_cast<double>(millimeters) / 25.4 * dpi));

    return pixels;
}

int Calculator::convertPixelsToPoints(const int& pixels, const int& dpi) {
    auto pixelsF = static_cast<double>(pixels);
    auto dpiF = static_cast<double>(dpi);
    auto points = static_cast<int>(std::round(pixelsF / dpiF * 72.0));

    return points;
}

int Calculator::convertPointsToPixels(const int& points, const int& dpi) {
    double inches = static_cast<double>(points) / 72.0;
    int pixels = static_cast<int>(std::round(inches * dpi));

    return pixels;
}

std::vector<std::array<int, 4>> Calculator::getObjectRectsInArea(
    const std::array<int, 2>& areaSize,
    const std::vector<std::array<int, 2>>& objectSizes,
    const std::array<int, 4>& margins,
    const std::array<int, 2>& spacings
    ) {
    auto getObjectAdjustedSizes = [](
        const std::vector<std::array<int, 2>>& givenObjectSizes,
        const double& ratio
        ) {
        std::vector<std::array<int, 2>> objectAdjustedSizes;

        for (auto& [objectWidth, objectHeight] : givenObjectSizes) {
            const auto objectWidthAdjusted = static_cast<int>(std::round(objectWidth * ratio));
            const auto objectHeightAdjusted = static_cast<int>(std::round(objectHeight * ratio));
            std::array<int, 2> objectAdjustedSize{objectWidthAdjusted, objectHeightAdjusted};
            objectAdjustedSizes.emplace_back(objectAdjustedSize);
        }

        return objectAdjustedSizes;
    };

    auto getObjectWidthSum = [](const std::vector<std::array<int, 4>>& givenObjectRects) {
        int objectWidthSum = 0;
        for (auto [objectX, objectY, objectWidth, objectHeight] : givenObjectRects) {
            objectWidthSum += objectWidth;
        }

        return objectWidthSum;
    };

    auto getObjectMaxHeight = [](const std::vector<std::array<int, 4>>& givenObjectRects) {
        std::vector<int> objectHeights;
        for (auto [objectX, objectY, objectWidth, objectHeight] : givenObjectRects) {
            objectHeights.emplace_back(objectHeight);
        }

        int objectMaxHeight = *std::ranges::max_element(objectHeights.begin(), objectHeights.end());

        return objectMaxHeight;
    };

    auto [topMargin, bottomMargin, leftMargin, rightMargin] = margins;
    auto [horizontalSpacing, verticalSpacing] = spacings;

    auto [areaWidth, areaHeight] = areaSize;
    const int areaUsableWidth = areaWidth - leftMargin - rightMargin;
    const int areaUsableHeight = areaHeight - topMargin - bottomMargin;

    std::vector<std::array<int, 2>> preprocessedObjectSizes;
    for (auto& objectSize : objectSizes) {
        auto [objectWidth, objectHeight]= objectSize;
        double ratioHeightToWidth = static_cast<double>(objectHeight) / static_cast<double>(objectWidth);

        int preprocessedObjectWidth = objectWidth;
        int preprocessedObjectHeight = objectHeight;

        if (preprocessedObjectWidth > areaUsableWidth) {
            preprocessedObjectWidth = areaUsableWidth;
            preprocessedObjectHeight = static_cast<int>(std::round(preprocessedObjectWidth * ratioHeightToWidth));
        }

        if (preprocessedObjectHeight > areaUsableHeight) {
            preprocessedObjectHeight = areaUsableHeight;
            preprocessedObjectWidth = static_cast<int>(
                std::round(static_cast<double>(preprocessedObjectHeight) / ratioHeightToWidth)
            );
        }

        std::array<int, 2> preprocessedObjectSize{preprocessedObjectWidth, preprocessedObjectHeight};
        preprocessedObjectSizes.emplace_back(preprocessedObjectSize);
    }

    const std::array<int, 2> initialPoint{leftMargin, areaHeight - topMargin};

    constexpr int totalTryingSteps = 50;
    std::vector<std::vector<std::array<int, 4>>> layout;

    for (int i = totalTryingSteps; i > 0; --i) {
        double ratio = static_cast<double>(i) / static_cast<double>(totalTryingSteps);
        auto objectAdjustedSizes = getObjectAdjustedSizes(preprocessedObjectSizes, ratio);

        layout = {};

        for (auto& objectSize : objectAdjustedSizes) {
            auto [objectWidth, objectHeight] = objectSize;

            if (layout.empty()) {
                std::vector<std::array<int, 4>> line;
                std::array<int, 4> objectRect{
                    initialPoint[0], initialPoint[1] - objectHeight, objectWidth, objectHeight
                };
                line.emplace_back(objectRect);
                layout.emplace_back(line);
            } else {
                auto& lastLine = layout.back();

                int lastLineAvailableWidth
                = static_cast<int>(
                    areaUsableWidth
                    - getObjectWidthSum(lastLine)
                    - horizontalSpacing * lastLine.size()
                );

                if (objectWidth <= lastLineAvailableWidth) {
                    auto [
                        lastObjectX,
                        lastObjectY,
                        lastObjectWidth,
                        lastObjectHeight
                        ] = lastLine.back();

                    std::array<int, 4> objectRect{
                        lastObjectX + lastObjectWidth + horizontalSpacing,
                        lastObjectY + lastObjectHeight - objectHeight,
                        objectWidth,
                        objectHeight
                    };

                    lastLine.emplace_back(objectRect);
                } else {
                    std::vector<std::array<int, 4>> newLine;

                    auto lastLineMaxHeight = getObjectMaxHeight(lastLine);

                    auto [
                        lastLineFirstObjectX,
                        lastLineFirstObjectY,
                        lastLineFirstObjectWidth,
                        lastLineFirstObjectHeight
                        ] = lastLine[0];

                    int objectY = lastLineFirstObjectY
                                  + lastLineFirstObjectHeight
                                  - lastLineMaxHeight
                                  - verticalSpacing
                                  - objectHeight;

                    std::array<int, 4> objectRect{
                        initialPoint[0],
                        objectY,
                        objectWidth,
                        objectHeight
                    };

                    newLine.emplace_back(objectRect);
                    layout.emplace_back(newLine);
                }
            }
        }

        int verticalHeight = 0;
        for (auto& line : layout) {
            auto lineMaxHeight = getObjectMaxHeight(line);
            verticalHeight += lineMaxHeight;
        }

        verticalHeight = static_cast<int>(verticalHeight + verticalSpacing * (layout.size() - 1));

        if (verticalHeight <= areaUsableHeight) {
            break;
        }
    }

    std::vector<std::array<int, 4>> objectRects = getCenteredObjectRects(layout);

    return objectRects;
}

std::vector<std::array<int, 4>> Calculator::getCenteredObjectRects(
    const std::vector<std::vector<std::array<int, 4>>>& layout
    ) {
    auto getObjectMaxHeight = [](const std::vector<std::array<int, 4>>& givenObjectRects) {
        std::vector<int> objectHeights;
        for (auto [objectX, objectY, objectWidth, objectHeight] : givenObjectRects) {
            objectHeights.emplace_back(objectHeight);
        }

        int objectMaxHeight = *std::ranges::max_element(objectHeights.begin(), objectHeights.end());

        return objectMaxHeight;
    };

    std::vector<std::array<int, 4>> objectRects;

    for (auto& line : layout) {
        auto objectMaxHeight = getObjectMaxHeight(line);

        for (auto& objectRect : line) {
            auto [objectX, objectY, objectWidth, objectHeight] = objectRect;

            if (objectHeight < objectMaxHeight) {
                int objectNewY = objectY - (objectMaxHeight - objectHeight) / 2;
                std::array<int, 4> objectNewRect{objectX, objectNewY, objectWidth, objectHeight};
                objectRects.emplace_back(objectNewRect);
            } else {
                objectRects.emplace_back(objectRect);
            }
        }
    }

    return objectRects;
}