#include "config.h"

#include <iostream>
#include <fstream>
#include <ranges>
#include <expected>
#include <algorithm>
#include <sstream>


/* The example of aiConnect.conf
 * keys=OpenAI:key1,Gemini:key2
 * models=Claude:Claude Sonnet 4:claude-sonnet-4-20250514,Mistral:Codestral 2508:codestral-2508
 * role=translator:You are a translator.:Requirements: ...
 * role=teacher:You are a teacher.:Requirements: ...
 */

std::vector<std::string> Config::supportedProviders = {"OpenAI", "Gemini", "Claude", "Mistral"};

std::filesystem::path Config::getConfPath() {
#ifdef _WIN32
    std::filesystem::path parentPath(std::getenv("LOCALAPPDATA"));
#else
    std::filesystem::path parentPath(std::getenv("HOME"));
    parentPath /= ".config";
#endif

    auto confFolderPath = parentPath / "aiConnect";

    if (!std::filesystem::exists(confFolderPath)) {
        std::filesystem::create_directories(confFolderPath);
    }

    auto confPath = confFolderPath / "aiConnect.conf";

    return confPath;
}

bool Config::saveConf(const std::string& optionName, const std::string& optionValue) {
    // Load all parameters from the conf file into maps.
    auto [
        keys,
        models,
        roles
        ] = loadConf("save");

    // Update maps with inputted data.
    if (optionName == "--add-key") {
        auto recognizedKeys = recognizeKeys(optionValue);
        if (recognizedKeys.empty()) return false;

        for (const auto& [provider, key] : recognizedKeys) {
            keys[provider] = key;
        }

    } else if (optionName == "--add-model") {
        auto recognizedModels = recognizeModels(optionValue);
        if (recognizedModels.empty()) return false;

        for (const auto& [provider, modelNames] : recognizedModels) {
            for (const auto& [customName, identifierName] : modelNames) {
                models[provider][customName] = identifierName;
            }
        }
    } else if (optionName == "--add-role") {
        auto recognizedRole = recognizeRole(optionValue, "save");
        if (recognizedRole.empty()) return false;

        for (const auto& [roleName, roleInstructions] : recognizedRole) {
            roles[roleName] = roleInstructions;
        }
    } else if (optionName == "--remove-key") {
        keys.erase(cleanString(optionValue));
    } else if (optionName == "--remove-model") {
        auto modelName = optionValue
        | std::views::split(':')
        | std::ranges::to<std::vector<std::string>>();

        if (modelName.size() == 2) {
            models[cleanString(modelName[0])].erase(cleanString(modelName[1]));
        }

    } else if (optionName == "--remove-role") {
        roles.erase(cleanString(optionValue));
    }

    // Prepare a formatted text that will be saved to the conf file.
    std::string conf;

    // Notice for the conf file.
    conf += "/* The conf file is automatically generated. "
            "Any contents other than default parameters will be lost. */\n\n";

    if (!keys.empty()) {
        conf += "keys=";
        for (const auto& [provider, key] : keys) {
            conf += provider;
            conf += ":";
            conf += key;
            conf += ",";
        }

        if (conf[conf.size() - 1] == ',') {
            conf.erase(conf.size() - 1, 1);
        }

        conf += "\n";
    }

    if (!models.empty()) {
        conf += "models=";
        for (const auto& [provider, modelNames] : models) {
            for (const auto& [customName, identifierName] : modelNames) {
                conf += provider;
                conf += ":";
                conf += customName;
                conf += ":";
                conf += identifierName;
                conf += ",";
            }
        }

        if (conf[conf.size() - 1] == ',') {
            conf.erase(conf.size() - 1, 1);
        }

        conf += "\n";
    }

    if (!roles.empty()) {
        for (const auto& [roleName, roleInstructions] : roles) {
            auto [roleInstruction, detailedInstructions] = roleInstructions;
            conf += "role=";
            conf += roleName;
            conf += ":";
            conf += roleInstruction;
            conf += ":";
            conf += detailedInstructions;
            conf += "\n";
        }
    }

    if (!conf.empty()) {
        // Override the conf file
        auto confPath = getConfPath();
        std::ofstream file(confPath);
        file << conf;

        return true;
    }

    return false;
}

std::tuple<
    std::unordered_map<std::string, std::string>,
    std::unordered_map<std::string, std::unordered_map<std::string, std::string>>,
    std::unordered_map<std::string, std::array<std::string, 2>>
>
Config::loadConf(const std::string& purpose) {
    auto confPath = getConfPath();

    auto getParameter = [](const std::string& line) -> std::expected<std::tuple<std::string, std::string>, int> {
        auto parameter = line
        | std::views::split('=')
        | std::ranges::to<std::vector<std::string>>();

        if (parameter.size() == 2 && !parameter[1].empty()) {
            return std::make_tuple(parameter[0], parameter[1]);
        }

        return std::unexpected(1);
    };

    std::ifstream file(confPath);

    std::string line;
    std::vector<std::string> configValues;
    std::unordered_map<std::string, std::string> keys;
    std::unordered_map<std::string, std::unordered_map<std::string, std::string>> models;
    std::unordered_map<std::string, std::array<std::string, 2>> roles;
    while (std::getline(file, line)) {
        auto parameter = getParameter(line);
        if (parameter.has_value()) {
            auto parameterName = std::get<0>(parameter.value());
            auto parameterValue = std::get<1>(parameter.value());

            if (parameterName == "keys") {
                keys = recognizeKeys(parameterValue);
            } else if (parameterName == "models") {
                models = recognizeModels(parameterValue);
            } else if (parameterName == "role") {
                auto role = recognizeRole(parameterValue, purpose);
                for (const auto& [roleName, roleInstructions] : role) {
                    roles[roleName] = roleInstructions;
                }
            }
        }
    }

    return std::make_tuple(keys, models, roles);
}

std::unordered_map<std::string, std::string> Config::recognizeKeys(const std::string& parameterValue) {
    auto commaVector =
            parameterValue
            | std::views::split(',')
            | std::ranges::to<std::vector<std::string>>();

    std::unordered_map<std::string, std::string> keys;

    for (auto& str: commaVector) {
        auto colonVector =
        str
        | std::views::split(':')
        | std::ranges::to<std::vector<std::string>>();

        if (std::ranges::contains(supportedProviders, colonVector[0]) && !colonVector[1].empty()) {
            keys[cleanString(colonVector[0])] = cleanString(colonVector[1]);
        }
    }

    return keys;
}

std::unordered_map<std::string, std::unordered_map<std::string, std::string>>
Config::recognizeModels(const std::string& parameterValue) {
    auto commaVector =
            parameterValue
            | std::views::split(',')
            | std::ranges::to<std::vector<std::string>>();

    std::unordered_map<std::string, std::unordered_map<std::string, std::string>> models;

    for (auto& str: commaVector) {
        auto colonVector =
        str
        | std::views::split(':')
        | std::ranges::to<std::vector<std::string>>();

        if (std::ranges::contains(supportedProviders, colonVector[0])
            && !colonVector[1].empty()
            && !colonVector[2].empty()
            ) {
            models[cleanString(colonVector[0])][cleanString(colonVector[1])] = cleanString(colonVector[2]);
        }
    }

    return models;
}

std::unordered_map<std::string, std::array<std::string, 2>>
Config::recognizeRole(std::string parameterValue, const std::string& purpose
    ) {
    std::stringstream stringStream(parameterValue);
    std::string stringPiece;

    std::string roleName;
    std::string roleInstruction;

    int i = 0;
    while (std::getline(stringStream, stringPiece, ':')) {
        if (i == 0) {
            roleName = stringPiece;
        } else if (i == 1) {
            roleInstruction = stringPiece;
            break;
        }

        ++i;
    }

    std::string detailedInstructionsRaw = parameterValue.erase(
        0, roleName.size() + roleInstruction.size() + 2);

    /* When inputting detailed instructions for a role, a line ends with \n.
     * The \n should be saved as \\n in .conf file, so the program can recognize
     * all lines correctly.
     * When sending detailed instructions to AI or showing detailed instructions,
     * \\n should be converted to \n for natural layout.
     */
    std::string detailedInstructions;
    for (int j = 0; j < detailedInstructionsRaw.size();) {
        if (purpose == "save"
            && detailedInstructionsRaw[j] == '\n'
            ) {
            detailedInstructions += '\\';
            detailedInstructions += '\\';
            detailedInstructions += 'n';
            ++j;
        } else if (purpose == "read"
            && detailedInstructionsRaw[j] == '\\'
            && detailedInstructionsRaw[j + 1] == '\\'
            && detailedInstructionsRaw[j + 2] == 'n'
            ) {
            detailedInstructions += '\n';
            j = j + 3;
        } else {
            detailedInstructions += detailedInstructionsRaw[j];
            ++j;
        }
    }

    std::unordered_map<std::string, std::array<std::string, 2>> role;

    if (!roleName.empty()) {
        role[cleanString(roleName)] = {cleanString(roleInstruction), cleanString(detailedInstructions)};
    }

    return role;
}

std::string Config::cleanString(std::string string) {
    while (string[0] == ' ') {
        string.erase(0, 1);
    }

    while (string[string.size() - 1] == ' ') {
        string.erase(string.size() - 1, 1);
    }

    if (string[0] == '"' && string[string.size() - 1] == '"') {
        string.erase(0, 1);
        string.erase(string.size() - 1, 1);
    }

    return string;
}