#include "net.h"

#include <filesystem>
#include <fstream>

#include <curl/curl.h>

#include "libJSONHandle/include/escape.h"
#include "libJSONHandle/include/dataquery.h"


std::expected<std::string, std::string> Net::aiDispatcher(
    const std::string& connectionProvider,
    const std::string& connectionKey,
    const std::string& connectionModel,
    std::string connectionRoleInstruction,
    std::string connectionDetailedInstructions,
    std::string connectionContent
    ) {
    std::tuple<std::string, std::vector<std::string>, std::string> apiParameters;

    connectionRoleInstruction = Escape::escapeCharacters(connectionRoleInstruction);
    connectionDetailedInstructions = Escape::escapeCharacters(connectionDetailedInstructions);
    connectionContent = Escape::escapeCharacters(connectionContent);

    std::string apiReplyQueryCondition;
    std::string apiReplyQuery;
    std::string apiReplyErrorInfo;

    if (connectionProvider == "OpenAI") {
        apiParameters = prepareOpenAIParameters(
            connectionKey,
            connectionModel,
            connectionRoleInstruction,
            connectionDetailedInstructions,
            connectionContent);

        apiReplyQueryCondition = "choices.message.role=assistant";
        apiReplyQuery = "choices.message.content";
        apiReplyErrorInfo = "error.message";
    } else if (connectionProvider == "Gemini") {
        apiParameters = prepareGeminiParameters(
            connectionKey,
            connectionModel,
            connectionRoleInstruction,
            connectionDetailedInstructions,
            connectionContent);

        apiReplyQueryCondition = "candidates.content.role=model";
        apiReplyQuery = "candidates.content.parts.text";
        apiReplyErrorInfo = "error.message";
    } else if (connectionProvider == "Claude") {
        apiParameters = prepareClaudeParameters(
            connectionKey,
            connectionModel,
            connectionRoleInstruction,
            connectionDetailedInstructions,
            connectionContent);

        apiReplyQueryCondition = "content.type=text";
        apiReplyQuery = "content.text";
        apiReplyErrorInfo = "error.message";
    } else if (connectionProvider == "Mistral") {
        apiParameters = prepareMistralParameters(
            connectionKey,
            connectionModel,
            connectionRoleInstruction,
            connectionDetailedInstructions,
            connectionContent);

        apiReplyQueryCondition = "choices.message.role=assistant";
        apiReplyQuery = "choices.message.content";
        apiReplyErrorInfo = "message";
    }

    auto [apiUrl, apiHeaders, jsonQuery] = apiParameters;

    if (!apiUrl.empty()
        && !apiHeaders.empty()
        && !jsonQuery.empty()
        && !apiReplyQuery.empty()
        && !apiReplyErrorInfo.empty()
        ) {
        auto reply = getReply(apiParameters);
        if (reply.has_value()) {
            auto replyValue = reply.value();

            /* Read data as a string. The string is a JSON text. */
            std::string replyJSON(replyValue.begin(), replyValue.end());

            /* replyPair contains two parts.
             * replyPair.first is the type of the query's result,
             * and replyPair.second is the query's result.
             * Here apiReplyQueryCondition may be replaced with an empty string "",
             * because the combination of query keys is unique.
             */
            std::pair<std::string, std::string> replyPair;
            replyPair = DataQuery::dataQuery(
                replyJSON, apiReplyQuery, apiReplyQueryCondition
            );

            /* If the query gets no result, the program tries to make a query
             * for error information.
             */
            if (replyPair.second.empty()) {
                replyPair = DataQuery::dataQuery(
                    replyJSON, apiReplyErrorInfo, ""
                );
            }

            /* Before the text is shown to the user, the program restores
             * newline characters \n and double quotation marks " from their
             * escaped characters.
             */
            std::string replyText = Escape::parseCharacters(replyPair.second);

            return replyText;
        }

        return std::unexpected(reply.error());
    }

    return std::unexpected("");
}

std::tuple<std::string, std::vector<std::string>, std::string>
Net::prepareOpenAIParameters(
    const std::string& connectionKey,
    const std::string& connectionModel,
    const std::string& connectionRoleInstruction,
    const std::string& connectionDetailedInstructions,
    const std::string& connectionContent
    ) {
    std::string apiUrl = "https://api.openai.com/v1/chat/completions";

    std::vector<std::string> apiHeaders;
    std::string apiKeyHeader = "Authorization: Bearer " + connectionKey;
    apiHeaders.emplace_back(apiKeyHeader);

    std::ostringstream jsonQuery;
    jsonQuery << '{'
                << '"' << "model"    << '"' << ": " << '"' << connectionModel << '"' << ", "
                << '"' << "messages" << '"' << ": " << '['
                  << '{'
                    << '"' << "role"    << '"' << ": " << '"' << "developer" << '"' << ", "
                    << '"' << "content" << '"' << ": " << '"' << connectionRoleInstruction << '"'
                  << '}' << ", "
                  << '{'
                    << '"' << "role"    << '"' << ": " << '"' << "user" << '"' << ", "
                    << '"' << "content" << '"' << ": " << '"' << connectionContent << '\\' << 'n'
                                                              << connectionDetailedInstructions << '"'
                  << '}'
                << ']'
              << '}';

    return std::make_tuple(apiUrl, apiHeaders, jsonQuery.str());
}

std::tuple<std::string, std::vector<std::string>, std::string>
Net::prepareGeminiParameters(
    const std::string& connectionKey,
    const std::string& connectionModel,
    const std::string& connectionRoleInstruction,
    const std::string& connectionDetailedInstructions,
    const std::string& connectionContent
    ) {
    std::string apiUrl = "https://generativelanguage.googleapis.com/v1beta/models/"
                         + connectionModel
                         + ":generateContent";

    std::vector<std::string> apiHeaders;
    std::string apiKeyHeader = "x-goog-api-key: " + connectionKey;
    apiHeaders.emplace_back(apiKeyHeader);

    std::ostringstream jsonQuery;
    jsonQuery << '{'
                << '"' << "contents" << '"' << ": " << '['
                  << '{'
                    << '"' << "parts" << '"' << ": " << '['
                      << '{'
                        << '"' << "text" << '"' << ": " << '"' << connectionContent << '\\' << 'n'
                                                               << connectionRoleInstruction << '\\' << 'n'
                                                               << connectionDetailedInstructions << '"'
                      << '}'
                    << ']'
                  << '}'
                << ']'
              << '}';

    return std::make_tuple(apiUrl, apiHeaders, jsonQuery.str());
}

std::tuple<std::string, std::vector<std::string>, std::string>
Net::prepareClaudeParameters(
    const std::string& connectionKey,
    const std::string& connectionModel,
    const std::string& connectionRoleInstruction,
    const std::string& connectionDetailedInstructions,
    const std::string& connectionContent
    ) {
    std::string apiUrl = "https://api.anthropic.com/v1/messages";

    std::vector<std::string> apiHeaders;
    std::string apiKeyHeader = "x-api-key: " + connectionKey;
    std::string apiVersionHeader = "anthropic-version: 2023-06-01";
    apiHeaders.emplace_back(apiKeyHeader);
    apiHeaders.emplace_back(apiVersionHeader);

    std::ostringstream jsonQuery;
    jsonQuery << '{'
                << '"' << "model"      << '"' << ": " << '"' << connectionModel << '"' << ", "
                << '"' << "max_tokens" << '"' << ": " << 1000 << ", "
                << '"' << "messages"   << '"' << ": " << '['
                  << '{'
                    << '"' << "role"    << '"' << ": " << '"' << "user" << '"' << ", "
                    << '"' << "content" << '"' << ": " << '"' << connectionContent << '\\' << 'n'
                                                              << connectionRoleInstruction << '\\' << 'n'
                                                              << connectionDetailedInstructions << '"'
                  << '}'
                << ']'
              << '}';

    return std::make_tuple(apiUrl, apiHeaders, jsonQuery.str());
}

std::tuple<std::string, std::vector<std::string>, std::string>
Net::prepareMistralParameters(
    const std::string& connectionKey,
    const std::string& connectionModel,
    const std::string& connectionRoleInstruction,
    const std::string& connectionDetailedInstructions,
    const std::string& connectionContent
    ) {
    std::string apiUrl = "https://api.mistral.ai/v1/chat/completions";

    std::vector<std::string> apiHeaders;
    std::string apiKeyHeader = "Authorization: Bearer " + connectionKey;
    apiHeaders.emplace_back(apiKeyHeader);

    std::ostringstream jsonQuery;
    jsonQuery << '{'
                << '"' << "model"    << '"' << ": " << '"' << connectionModel << '"' << ", "
                << '"' << "messages" << '"' << ": " << '['
                  << '{'
                    << '"' << "role"    << '"' << ": " << '"' << "user" << '"' << ", "
                    << '"' << "content" << '"' << ": " << '"' << connectionContent << '\\' << 'n'
                                                              << connectionRoleInstruction << '\\' << 'n'
                                                              << connectionDetailedInstructions << '"'
                  << '}'
                << ']'
              << '}';

    return std::make_tuple(apiUrl, apiHeaders, jsonQuery.str());
}

std::expected<std::vector<uint8_t>, std::string>
Net::getReply(
    const std::tuple<std::string, std::vector<std::string>, std::string>& apiParameters
    ) {
    CURL *curl = curl_easy_init();
    if (curl) {
        auto [apiUrl, apiHeaders, jsonQuery] = apiParameters;

        curl_easy_setopt(curl, CURLOPT_URL, apiUrl.c_str());

        struct curl_slist *header = nullptr;
        header = curl_slist_append(header, "Content-Type: application/json");

        for (auto& apiHeader : apiHeaders) {
            header = curl_slist_append(header, apiHeader.c_str());
        }

        curl_easy_setopt(curl, CURLOPT_HTTPHEADER, header);

        curl_easy_setopt(curl, CURLOPT_POSTFIELDS, jsonQuery.c_str());

        std::vector<uint8_t> receivedData;

        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, curlWriteCallback);
        curl_easy_setopt(curl, CURLOPT_WRITEDATA, &receivedData);

        CURLcode res = curl_easy_perform(curl);

        std::ostringstream errorInfo;
        if (res != CURLE_OK) {
            errorInfo << "The connection failed, and the response code is " << res;
        }

        curl_slist_free_all(header);
        curl_easy_cleanup(curl);

        if (!receivedData.empty()) {
            return receivedData;
        }

        return std::unexpected(errorInfo.str());
    }

    return std::unexpected("Failed to start a connection.");
}

size_t Net::curlWriteCallback(char* data, size_t size, size_t nmemb, void* userdata) {
    auto receivedSize = size * nmemb;
    auto* receivedData = static_cast<std::vector<uint8_t>*>(userdata);
    receivedData->insert(receivedData->end(), data, data + receivedSize);

    return receivedSize;
}
